<?php
/**
 * Plugin Name: BOG Payment Gateway
 * Plugin URI: https://function.ge
 * Description: Bank of Georgia credit card payment gateway for WordPress
 * Version: 1.0.0
 * Author: function.ge
 * Author URI: https://function.ge
 * Text Domain: bog-payment-gateway
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('BOG_PAYMENT_VERSION', '1.0.0');
define('BOG_PAYMENT_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('BOG_PAYMENT_PLUGIN_URL', plugin_dir_url(__FILE__));
define('BOG_PAYMENT_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Include required files (only non-WooCommerce dependent classes)
require_once BOG_PAYMENT_PLUGIN_DIR . 'includes/class-bog-payment-api.php';
require_once BOG_PAYMENT_PLUGIN_DIR . 'includes/class-bog-payment-callback.php';
require_once BOG_PAYMENT_PLUGIN_DIR . 'includes/class-bog-payment-admin.php';

/**
 * Main plugin class
 */
class BOG_Payment_Gateway_Plugin {
    
    /**
     * Single instance
     */
    private static $instance = null;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Initialize plugin
     */
    public function init() {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            add_action('admin_notices', array($this, 'woocommerce_missing_notice'));
            return;
        }
        
        // Now it's safe to load the gateway class
        require_once BOG_PAYMENT_PLUGIN_DIR . 'includes/class-bog-payment-gateway.php';
        
        // Initialize components
        BOG_Payment_Callback::get_instance();
        BOG_Payment_Admin::get_instance();
        
        // Add payment gateway to WooCommerce
        add_filter('woocommerce_payment_gateways', array($this, 'add_gateway'));
    }
    
    /**
     * Add gateway to WooCommerce
     */
    public function add_gateway($gateways) {
        $gateways[] = 'BOG_Payment_Gateway';
        return $gateways;
    }
    
    /**
     * WooCommerce missing notice
     */
    public function woocommerce_missing_notice() {
        echo '<div class="error"><p><strong>' . 
             esc_html__('BOG Payment Gateway requires WooCommerce to be installed and active.', 'bog-payment-gateway') . 
             '</strong></p></div>';
    }
    
    /**
     * Activate plugin
     */
    public function activate() {
        // Create database tables if needed
        global $wpdb;
        $charset_collate = $wpdb->get_charset_collate();
        
        $table_name = $wpdb->prefix . 'bog_payment_logs';
        
        $sql = "CREATE TABLE IF NOT EXISTS $table_name (
            id bigint(20) NOT NULL AUTO_INCREMENT,
            order_id varchar(100) NOT NULL,
            wc_order_id bigint(20) DEFAULT NULL,
            transaction_type varchar(50) NOT NULL,
            request_data longtext,
            response_data longtext,
            status varchar(50) NOT NULL,
            created_at datetime DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (id),
            KEY order_id (order_id),
            KEY wc_order_id (wc_order_id),
            KEY created_at (created_at)
        ) $charset_collate;";
        
        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Deactivate plugin
     */
    public function deactivate() {
        flush_rewrite_rules();
    }
}

// Initialize plugin
BOG_Payment_Gateway_Plugin::get_instance();
