<?php
/**
 * BOG Payment Admin Interface
 */

if (!defined('ABSPATH')) {
    exit;
}

class BOG_Payment_Admin {
    
    /**
     * Single instance
     */
    private static $instance = null;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        add_filter('plugin_action_links_' . BOG_PAYMENT_PLUGIN_BASENAME, array($this, 'add_action_links'));
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __('BOG Payment Logs', 'bog-payment-gateway'),
            __('BOG Payments', 'bog-payment-gateway'),
            'manage_woocommerce',
            'bog-payment-logs',
            array($this, 'render_logs_page')
        );
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ('woocommerce_page_bog-payment-logs' !== $hook) {
            return;
        }
        
        wp_enqueue_style('bog-payment-admin', BOG_PAYMENT_PLUGIN_URL . 'assets/css/admin.css', array(), BOG_PAYMENT_VERSION);
        wp_enqueue_script('bog-payment-admin', BOG_PAYMENT_PLUGIN_URL . 'assets/js/admin.js', array('jquery'), BOG_PAYMENT_VERSION, true);
    }
    
    /**
     * Add action links
     */
    public function add_action_links($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=wc-settings&tab=checkout&section=bog_payment') . '">' . 
                         __('Settings', 'bog-payment-gateway') . '</a>';
        $logs_link = '<a href="' . admin_url('admin.php?page=bog-payment-logs') . '">' . 
                     __('Logs', 'bog-payment-gateway') . '</a>';
        
        array_unshift($links, $settings_link, $logs_link);
        return $links;
    }
    
    /**
     * Render logs page
     */
    public function render_logs_page() {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'bog_payment_logs';
        
        // Handle actions
        if (isset($_POST['action']) && $_POST['action'] === 'clear_logs' && check_admin_referer('bog_clear_logs')) {
            $table_name = $wpdb->prefix . 'bog_payment_logs';
            // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
            $wpdb->query("TRUNCATE TABLE `{$table_name}`");
            echo '<div class="notice notice-success"><p>' . esc_html__('Logs cleared successfully.', 'bog-payment-gateway') . '</p></div>';
        }
        
        // Pagination
        $page = isset($_GET['paged']) ? max(1, intval($_GET['paged'])) : 1;
        $per_page = 50;
        $offset = ($page - 1) * $per_page;
        
        // Filters
        $where_clauses = array('1=1');
        $where_values = array();
        
        if (isset($_GET['status']) && $_GET['status'] !== '') {
            $where_clauses[] = 'status = %s';
            $where_values[] = sanitize_text_field(wp_unslash($_GET['status']));
        }
        if (isset($_GET['type']) && $_GET['type'] !== '') {
            $where_clauses[] = 'transaction_type = %s';
            $where_values[] = sanitize_text_field(wp_unslash($_GET['type']));
        }
        if (isset($_GET['wc_order_id']) && $_GET['wc_order_id'] !== '') {
            $where_clauses[] = 'wc_order_id = %d';
            $where_values[] = intval($_GET['wc_order_id']);
        }
        
        $where_sql = implode(' AND ', $where_clauses);
        $table_name = $wpdb->prefix . 'bog_payment_logs';
        
        // Build query with proper escaping
        if (count($where_values) > 0) {
            $placeholders = array_fill(0, count($where_values), '%s');
            $where_prepared = vsprintf(str_replace(array('status = %s', 'transaction_type = %s', 'wc_order_id = %d'), $placeholders, $where_sql), $where_values);
            $where_sql = $wpdb->prepare($where_prepared, $where_values); // phpcs:ignore WordPress.DB.PreparedSQL.NotPrepared
        }
        
        // Get logs
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $logs = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT * FROM `{$table_name}` WHERE {$where_sql} ORDER BY created_at DESC LIMIT %d OFFSET %d",
                $per_page,
                $offset
            )
        );
        
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $total_logs = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT COUNT(*) FROM `{$table_name}` WHERE {$where_sql}"
            )
        );
        $total_pages = ceil($total_logs / $per_page);
        
        // Get statistics
        // phpcs:ignore WordPress.DB.DirectDatabaseQuery.DirectQuery, WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.PreparedSQL.InterpolatedNotPrepared
        $stats = array(
            'total' => $wpdb->get_var("SELECT COUNT(*) FROM `{$table_name}`"),
            'success' => $wpdb->get_var("SELECT COUNT(*) FROM `{$table_name}` WHERE status = 'success'"),
            'error' => $wpdb->get_var("SELECT COUNT(*) FROM `{$table_name}` WHERE status = 'error'"),
            'today' => $wpdb->get_var("SELECT COUNT(*) FROM `{$table_name}` WHERE DATE(created_at) = CURDATE()"),
        );
        
        ?>
        <div class="wrap">
            <h1><?php esc_html_e('BOG Payment Logs', 'bog-payment-gateway'); ?></h1>
            
            <div class="bog-stats">
                <div class="bog-stat-box">
                    <h3><?php echo number_format($stats['total']); ?></h3>
                    <p><?php esc_html_e('Total Transactions', 'bog-payment-gateway'); ?></p>
                </div>
                <div class="bog-stat-box">
                    <h3><?php echo number_format($stats['success']); ?></h3>
                    <p><?php esc_html_e('Successful', 'bog-payment-gateway'); ?></p>
                </div>
                <div class="bog-stat-box">
                    <h3><?php echo number_format($stats['error']); ?></h3>
                    <p><?php esc_html_e('Failed', 'bog-payment-gateway'); ?></p>
                </div>
                <div class="bog-stat-box">
                    <h3><?php echo number_format($stats['today']); ?></h3>
                    <p><?php esc_html_e('Today', 'bog-payment-gateway'); ?></p>
                </div>
            </div>
            
            <div class="bog-filters">
                <form method="get">
                    <input type="hidden" name="page" value="bog-payment-logs">
                    
                    <select name="status">
                        <option value=""><?php esc_html_e('All Statuses', 'bog-payment-gateway'); ?></option>
                        <option value="success" <?php selected(isset($_GET['status']) ? sanitize_text_field(wp_unslash($_GET['status'])) : '', 'success'); ?>><?php esc_html_e('Success', 'bog-payment-gateway'); ?></option>
                        <option value="error" <?php selected(isset($_GET['status']) ? sanitize_text_field(wp_unslash($_GET['status'])) : '', 'error'); ?>><?php esc_html_e('Error', 'bog-payment-gateway'); ?></option>
                    </select>
                    
                    <select name="type">
                        <option value=""><?php esc_html_e('All Types', 'bog-payment-gateway'); ?></option>
                        <option value="create_order" <?php selected(isset($_GET['type']) ? sanitize_text_field(wp_unslash($_GET['type'])) : '', 'create_order'); ?>><?php esc_html_e('Create Order', 'bog-payment-gateway'); ?></option>
                        <option value="callback" <?php selected(isset($_GET['type']) ? sanitize_text_field(wp_unslash($_GET['type'])) : '', 'callback'); ?>><?php esc_html_e('Callback', 'bog-payment-gateway'); ?></option>
                        <option value="refund" <?php selected(isset($_GET['type']) ? sanitize_text_field(wp_unslash($_GET['type'])) : '', 'refund'); ?>><?php esc_html_e('Refund', 'bog-payment-gateway'); ?></option>
                        <option value="get_payment_details" <?php selected(isset($_GET['type']) ? sanitize_text_field(wp_unslash($_GET['type'])) : '', 'get_payment_details'); ?>><?php esc_html_e('Get Details', 'bog-payment-gateway'); ?></option>
                    </select>
                    
                    <input type="number" name="wc_order_id" placeholder="<?php esc_attr_e('WC Order ID', 'bog-payment-gateway'); ?>" value="<?php echo isset($_GET['wc_order_id']) ? esc_attr(sanitize_text_field(wp_unslash($_GET['wc_order_id']))) : ''; ?>">
                    
                    <button type="submit" class="button"><?php esc_html_e('Filter', 'bog-payment-gateway'); ?></button>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=bog-payment-logs')); ?>" class="button"><?php esc_html_e('Reset', 'bog-payment-gateway'); ?></a>
                </form>
                
                <form method="post" style="display: inline-block;">
                    <?php wp_nonce_field('bog_clear_logs'); ?>
                    <input type="hidden" name="action" value="clear_logs">
                    <button type="submit" class="button button-secondary" onclick="return confirm('<?php echo esc_js(__('Are you sure you want to clear all logs?', 'bog-payment-gateway')); ?>')">
                        <?php esc_html_e('Clear All Logs', 'bog-payment-gateway'); ?>
                    </button>
                </form>
            </div>
            
            <table class="wp-list-table widefat fixed striped">
                <thead>
                    <tr>
                        <th><?php esc_html_e('ID', 'bog-payment-gateway'); ?></th>
                        <th><?php esc_html_e('BOG Order ID', 'bog-payment-gateway'); ?></th>
                        <th><?php esc_html_e('WC Order', 'bog-payment-gateway'); ?></th>
                        <th><?php esc_html_e('Type', 'bog-payment-gateway'); ?></th>
                        <th><?php esc_html_e('Status', 'bog-payment-gateway'); ?></th>
                        <th><?php esc_html_e('Date', 'bog-payment-gateway'); ?></th>
                        <th><?php esc_html_e('Actions', 'bog-payment-gateway'); ?></th>
                    </tr>
                </thead>
                <tbody>
                    <?php if ($logs): ?>
                        <?php foreach ($logs as $log): ?>
                            <tr>
                                <td><?php echo esc_html($log->id); ?></td>
                                <td><?php echo esc_html($log->order_id); ?></td>
                                <td>
                                    <?php if ($log->wc_order_id): ?>
                                        <a href="<?php echo esc_url(admin_url('post.php?post=' . $log->wc_order_id . '&action=edit')); ?>" target="_blank">
                                            #<?php echo esc_html($log->wc_order_id); ?>
                                        </a>
                                    <?php else: ?>
                                        -
                                    <?php endif; ?>
                                </td>
                                <td><?php echo esc_html($log->transaction_type); ?></td>
                                <td>
                                    <span class="bog-status bog-status-<?php echo esc_attr($log->status); ?>">
                                        <?php echo esc_html($log->status); ?>
                                    </span>
                                </td>
                                <td><?php echo esc_html(mysql2date('Y-m-d H:i:s', $log->created_at)); ?></td>
                                <td>
                                    <button class="button button-small bog-view-details" data-log-id="<?php echo esc_attr($log->id); ?>">
                                        <?php esc_html_e('View Details', 'bog-payment-gateway'); ?>
                                    </button>
                                    
                                    <div id="bog-details-<?php echo esc_attr($log->id); ?>" style="display: none;">
                                        <h3><?php esc_html_e('Request Data', 'bog-payment-gateway'); ?></h3>
                                        <pre><?php echo esc_html($log->request_data); ?></pre>
                                        
                                        <h3><?php esc_html_e('Response Data', 'bog-payment-gateway'); ?></h3>
                                        <pre><?php echo esc_html($log->response_data); ?></pre>
                                    </div>
                                </td>
                            </tr>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <tr>
                            <td colspan="7"><?php esc_html_e('No logs found.', 'bog-payment-gateway'); ?></td>
                        </tr>
                    <?php endif; ?>
                </tbody>
            </table>
            
            <?php if ($total_pages > 1): ?>
                <div class="tablenav">
                    <div class="tablenav-pages">
                        <?php
                        // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- paginate_links is safe
                        echo paginate_links(array(
                            'base' => add_query_arg('paged', '%#%'),
                            'format' => '',
                            'prev_text' => __('&laquo;', 'bog-payment-gateway'),
                            'next_text' => __('&raquo;', 'bog-payment-gateway'),
                            'total' => $total_pages,
                            'current' => $page
                        ));
                        ?>
                    </div>
                </div>
            <?php endif; ?>
        </div>
        <?php
    }
}
