<?php
/**
 * BOG Payment API Class
 * Handles all API communications with Bank of Georgia
 */

if (!defined('ABSPATH')) {
    exit;
}

class BOG_Payment_API {
    
    /**
     * API endpoints
     */
    private $api_base_url;
    private $auth_url;
    
    /**
     * Credentials
     */
    private $client_id;
    private $client_secret;
    
    /**
     * Access token
     */
    private $access_token;
    private $token_expiry;
    
    /**
     * Test mode
     */
    private $test_mode;
    
    /**
     * Constructor
     */
    public function __construct($client_id, $client_secret, $test_mode = false) {
        $this->client_id = $client_id;
        $this->client_secret = $client_secret;
        $this->test_mode = $test_mode;
        
        // Set API URLs based on mode
        if ($test_mode) {
            $this->api_base_url = 'https://api.bog.ge/payments/v1';
            $this->auth_url = 'https://oauth2.bog.ge/auth/realms/bog/protocol/openid-connect/token';
        } else {
            $this->api_base_url = 'https://api.bog.ge/payments/v1';
            $this->auth_url = 'https://oauth2.bog.ge/auth/realms/bog/protocol/openid-connect/token';
        }
    }
    
    /**
     * Get access token (OAuth 2.0)
     */
    public function get_access_token() {
        // Check if we have a valid token
        if ($this->access_token && $this->token_expiry && time() < $this->token_expiry) {
            return $this->access_token;
        }
        
        // Request new token
        $response = wp_remote_post($this->auth_url, array(
            'headers' => array(
                'Content-Type' => 'application/x-www-form-urlencoded',
            ),
            'body' => array(
                'grant_type' => 'client_credentials',
                'client_id' => $this->client_id,
                'client_secret' => $this->client_secret,
            ),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            $this->log_error('Authentication failed', $response->get_error_message());
            return false;
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        
        if (isset($body['access_token'])) {
            $this->access_token = $body['access_token'];
            $this->token_expiry = time() + ($body['expires_in'] ?? 3600) - 60; // Subtract 60 seconds for safety
            return $this->access_token;
        }
        
        $this->log_error('Authentication failed', $body);
        return false;
    }
    
    /**
     * Create order
     */
    public function create_order($order_data) {
        $token = $this->get_access_token();
        if (!$token) {
            return array('success' => false, 'error' => 'Authentication failed');
        }
        
        $endpoint = $this->api_base_url . '/ecommerce/orders';
        
        $response = wp_remote_post($endpoint, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $token,
                'Accept-Language' => 'ka',
            ),
            'body' => json_encode($order_data),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->log_transaction('create_order', $order_data, array('error' => $error_message), 'error');
            return array('success' => false, 'error' => $error_message);
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        $this->log_transaction('create_order', $order_data, $body, $status_code == 201 ? 'success' : 'error');
        
        if ($status_code == 201 && isset($body['id'])) {
            return array(
                'success' => true,
                'order_id' => $body['id'],
                'redirect_url' => $body['_links']['redirect']['href'] ?? '',
                'details_url' => $body['_links']['details']['href'] ?? '',
            );
        }
        
        return array('success' => false, 'error' => $body['message'] ?? 'Order creation failed', 'data' => $body);
    }
    
    /**
     * Get payment details
     */
    public function get_payment_details($order_id) {
        $token = $this->get_access_token();
        if (!$token) {
            return array('success' => false, 'error' => 'Authentication failed');
        }
        
        $endpoint = $this->api_base_url . '/payment/status/' . $order_id;
        
        $response = wp_remote_get($endpoint, array(
            'headers' => array(
                'Authorization' => 'Bearer ' . $token,
                'Accept-Language' => 'ka',
            ),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->log_transaction('get_payment_details', array('order_id' => $order_id), array('error' => $error_message), 'error');
            return array('success' => false, 'error' => $error_message);
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        $this->log_transaction('get_payment_details', array('order_id' => $order_id), $body, 'success');
        
        if ($status_code == 200) {
            return array('success' => true, 'data' => $body);
        }
        
        return array('success' => false, 'error' => 'Failed to get payment details', 'data' => $body);
    }
    
    /**
     * Refund payment
     */
    public function refund_payment($order_id, $amount = null) {
        $token = $this->get_access_token();
        if (!$token) {
            return array('success' => false, 'error' => 'Authentication failed');
        }
        
        $endpoint = $this->api_base_url . '/payment/refund/' . $order_id;
        
        $body_data = array();
        if ($amount !== null) {
            $body_data['amount'] = (float) $amount;
        }
        
        $response = wp_remote_post($endpoint, array(
            'headers' => array(
                'Content-Type' => 'application/json',
                'Authorization' => 'Bearer ' . $token,
                'Idempotency-Key' => uniqid('refund_', true),
            ),
            'body' => json_encode($body_data),
            'timeout' => 30,
        ));
        
        if (is_wp_error($response)) {
            $error_message = $response->get_error_message();
            $this->log_transaction('refund', array('order_id' => $order_id, 'amount' => $amount), array('error' => $error_message), 'error');
            return array('success' => false, 'error' => $error_message);
        }
        
        $body = json_decode(wp_remote_retrieve_body($response), true);
        $status_code = wp_remote_retrieve_response_code($response);
        
        $this->log_transaction('refund', array('order_id' => $order_id, 'amount' => $amount), $body, $status_code == 200 ? 'success' : 'error');
        
        if ($status_code == 200) {
            return array('success' => true, 'data' => $body);
        }
        
        return array('success' => false, 'error' => $body['message'] ?? 'Refund failed', 'data' => $body);
    }
    
    /**
     * Log transaction
     */
    private function log_transaction($type, $request, $response, $status) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'bog_payment_logs';
        
        $wpdb->insert(
            $table_name,
            array(
                'order_id' => $request['external_order_id'] ?? ($request['order_id'] ?? ''),
                'transaction_type' => $type,
                'request_data' => json_encode($request),
                'response_data' => json_encode($response),
                'status' => $status,
                'created_at' => current_time('mysql'),
            ),
            array('%s', '%s', '%s', '%s', '%s', '%s')
        );
    }
    
    /**
     * Log error
     */
    private function log_error($message, $data = null) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BOG Payment API Error: ' . $message . ' - ' . print_r($data, true));
        }
    }
}
