<?php
/**
 * BOG Payment Callback Handler
 */

if (!defined('ABSPATH')) {
    exit;
}

class BOG_Payment_Callback {
    
    /**
     * Single instance
     */
    private static $instance = null;
    
    /**
     * Get instance
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        // Constructor intentionally left empty
    }
    
    /**
     * Process callback from BOG
     */
    public function process_callback($callback_data, $order) {
        // Log callback received (for debugging)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BOG Payment Callback Received - Order #' . $order->get_id());
            error_log('BOG Callback Data: ' . print_r($callback_data, true));
        }
        
        // Validate callback data
        if (!isset($callback_data['order_id']) || !isset($callback_data['order_status'])) {
            $this->log_error('Invalid callback data', $callback_data);
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('BOG Payment Callback Error: Invalid callback data structure');
            }
            return false;
        }
        
        $bog_order_id = $callback_data['order_id'];
        $status = $callback_data['order_status']['key'];
        $status_text = $callback_data['order_status']['value'] ?? '';
        
        // Update order meta with BOG order ID if not set
        if (!$order->get_meta('_bog_order_id')) {
            $order->update_meta_data('_bog_order_id', $bog_order_id);
        }
        
        // Save callback data
        $order->update_meta_data('_bog_callback_data', $callback_data);
        $order->save();
        
        // Process based on status
        switch ($status) {
            case 'created':
                $this->handle_created($order, $callback_data);
                break;
                
            case 'completed':
                $this->handle_completed($order, $callback_data);
                break;
                
            case 'blocked':
                $this->handle_blocked($order, $callback_data);
                break;
                
            case 'partial_completed':
                $this->handle_partial_completed($order, $callback_data);
                break;
                
            case 'rejected':
                $this->handle_rejected($order, $callback_data);
                break;
                
            case 'refunded':
                $this->handle_refunded($order, $callback_data);
                break;
                
            case 'expired':
                $this->handle_expired($order, $callback_data);
                break;
                
            default:
                $this->log_error('Unknown status', $callback_data);
                $order->add_order_note(
                    sprintf(
                        /* translators: %s: Payment status */
                        __('BOG Payment callback received with unknown status: %s', 'bog-payment-gateway'),
                        $status
                    )
                );
                break;
        }
        
        return true;
    }
    
    /**
     * Handle created status
     */
    private function handle_created($order, $data) {
        $order->add_order_note(__('BOG Payment created and pending.', 'bog-payment-gateway'));
    }
    
    /**
     * Handle completed status
     */
    private function handle_completed($order, $data) {
        // Check if already completed
        if ($order->has_status('completed') || $order->has_status('processing')) {
            return;
        }
        
        // Get payment details
        $transfer_amount = $data['purchase_units']['transfer_amount'] ?? 0;
        $transaction_id = $data['order_id'];
        
        // Update order
        $order->payment_complete($transaction_id);
        
        // Add order note
        $note = sprintf(
            /* translators: 1: Payment amount, 2: Transaction ID */
            __('BOG Payment completed. Amount: %1$s, Transaction ID: %2$s', 'bog-payment-gateway'),
            wc_price($transfer_amount, array('currency' => $order->get_currency())),
            $transaction_id
        );
        
        if (isset($data['payment_method'])) {
            $note .= sprintf(
                /* translators: %s: Payment method name */
                __(' Payment method: %s', 'bog-payment-gateway'),
                $this->get_payment_method_name($data['payment_method'])
            );
        }
        
        $order->add_order_note($note);
        
        // Save transaction details
        $order->update_meta_data('_bog_transaction_id', $transaction_id);
        $order->update_meta_data('_bog_transfer_amount', $transfer_amount);
        $order->update_meta_data('_bog_payment_method', $data['payment_method'] ?? '');
        $order->save();
    }
    
    /**
     * Handle blocked status (pre-authorization)
     */
    private function handle_blocked($order, $data) {
        $order->update_status('on-hold', __('BOG Payment: Amount blocked (pre-authorization). Awaiting confirmation.', 'bog-payment-gateway'));
        
        $blocked_amount = $data['purchase_units']['transfer_amount'] ?? 0;
        $order->add_order_note(
            sprintf(
                /* translators: %s: Blocked amount */
                __('BOG Payment: Amount %s blocked for pre-authorization.', 'bog-payment-gateway'),
                wc_price($blocked_amount, array('currency' => $order->get_currency()))
            )
        );
    }
    
    /**
     * Handle partial completed status
     */
    private function handle_partial_completed($order, $data) {
        $transfer_amount = $data['purchase_units']['transfer_amount'] ?? 0;
        $request_amount = $data['purchase_units']['request_amount'] ?? 0;
        
        $order->update_status('on-hold');
        $order->add_order_note(
            sprintf(
                /* translators: 1: Paid amount, 2: Total amount */
                __('BOG Payment: Partial payment received. Paid: %1$s of %2$s', 'bog-payment-gateway'),
                wc_price($transfer_amount, array('currency' => $order->get_currency())),
                wc_price($request_amount, array('currency' => $order->get_currency()))
            )
        );
    }
    
    /**
     * Handle rejected status
     */
    private function handle_rejected($order, $data) {
        $reason = $data['reject_reason']['message'] ?? __('Unknown reason', 'bog-payment-gateway');
        $order->update_status(
            'failed',
            sprintf(
                /* translators: %s: Rejection reason */
                __('BOG Payment rejected: %s', 'bog-payment-gateway'),
                $reason
            )
        );
    }
    
    /**
     * Handle refunded status
     */
    private function handle_refunded($order, $data) {
        $refund_amount = $data['purchase_units']['refund_amount'] ?? 0;
        
        $order->add_order_note(
            sprintf(
                /* translators: %s: Refunded amount */
                __('BOG Payment refunded: %s', 'bog-payment-gateway'),
                wc_price($refund_amount, array('currency' => $order->get_currency()))
            )
        );
    }
    
    /**
     * Handle expired status
     */
    private function handle_expired($order, $data) {
        $order->update_status('cancelled', __('BOG Payment expired.', 'bog-payment-gateway'));
    }
    
    /**
     * Get payment method name
     */
    private function get_payment_method_name($method) {
        $methods = array(
            'card' => __('Credit/Debit Card', 'bog-payment-gateway'),
            'google_pay' => __('Google Pay', 'bog-payment-gateway'),
            'apple_pay' => __('Apple Pay', 'bog-payment-gateway'),
            'bog_p2p' => __('BOG P2P', 'bog-payment-gateway'),
            'bog_loyalty' => __('BOG Loyalty', 'bog-payment-gateway'),
            'bog_loan' => __('BOG Loan', 'bog-payment-gateway'),
            'bnpl' => __('Buy Now Pay Later', 'bog-payment-gateway'),
        );
        
        return $methods[$method] ?? $method;
    }
    
    /**
     * Log error
     */
    private function log_error($message, $data = null) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BOG Payment Callback Error: ' . $message . ' - ' . print_r($data, true));
        }
    }
}
