<?php
/**
 * BOG Payment Gateway for WooCommerce
 */

if (!defined('ABSPATH')) {
    exit;
}

class BOG_Payment_Gateway extends WC_Payment_Gateway {
    
    /**
     * Constructor
     */
    public function __construct() {
        $this->id = 'bog_payment';
        $this->icon = '';
        $this->has_fields = false;
        $this->method_title = __('BOG Payment Gateway', 'bog-payment-gateway');
        $this->method_description = __('Accept credit card payments via Bank of Georgia', 'bog-payment-gateway');

        // Supports refunds (full and partial)
        $this->supports = array(
            'products',
            'refunds',
        );
        
        // Load settings
        $this->init_form_fields();
        $this->init_settings();
        
        // Get settings
        $this->title = $this->get_option('title');
        $this->description = $this->get_option('description');
        $this->enabled = $this->get_option('enabled');
        $this->testmode = 'yes' === $this->get_option('testmode');
        $this->client_id = $this->testmode ? $this->get_option('test_client_id') : $this->get_option('client_id');
        $this->client_secret = $this->testmode ? $this->get_option('test_client_secret') : $this->get_option('client_secret');
        
        // Actions
        add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
        add_action('woocommerce_api_bog_payment_callback', array($this, 'handle_callback'));
    }
    
    /**
     * Initialize form fields
     */
    public function init_form_fields() {
        $this->form_fields = array(
            'enabled' => array(
                'title' => __('Enable/Disable', 'bog-payment-gateway'),
                'type' => 'checkbox',
                'label' => __('Enable BOG Payment Gateway', 'bog-payment-gateway'),
                'default' => 'no',
            ),
            'title' => array(
                'title' => __('Title', 'bog-payment-gateway'),
                'type' => 'text',
                'description' => __('Payment method title that customers will see during checkout.', 'bog-payment-gateway'),
                'default' => __('Credit Card (BOG)', 'bog-payment-gateway'),
                'desc_tip' => true,
            ),
            'description' => array(
                'title' => __('Description', 'bog-payment-gateway'),
                'type' => 'textarea',
                'description' => __('Payment method description that customers will see during checkout.', 'bog-payment-gateway'),
                'default' => __('Pay securely with your credit card via Bank of Georgia.', 'bog-payment-gateway'),
                'desc_tip' => true,
            ),
            'testmode' => array(
                'title' => __('Test Mode', 'bog-payment-gateway'),
                'type' => 'checkbox',
                'label' => __('Enable Test Mode', 'bog-payment-gateway'),
                'default' => 'yes',
                'description' => __('Use test API credentials for testing.', 'bog-payment-gateway'),
            ),
            'client_id' => array(
                'title' => __('Client ID', 'bog-payment-gateway'),
                'type' => 'text',
                'description' => __('Your BOG API Client ID (Production)', 'bog-payment-gateway'),
                'default' => '',
                'desc_tip' => true,
            ),
            'client_secret' => array(
                'title' => __('Client Secret', 'bog-payment-gateway'),
                'type' => 'password',
                'description' => __('Your BOG API Client Secret (Production)', 'bog-payment-gateway'),
                'default' => '',
                'desc_tip' => true,
            ),
            'test_client_id' => array(
                'title' => __('Test Client ID', 'bog-payment-gateway'),
                'type' => 'text',
                'description' => __('Your BOG API Client ID (Test)', 'bog-payment-gateway'),
                'default' => '',
                'desc_tip' => true,
            ),
            'test_client_secret' => array(
                'title' => __('Test Client Secret', 'bog-payment-gateway'),
                'type' => 'password',
                'description' => __('Your BOG API Client Secret (Test)', 'bog-payment-gateway'),
                'default' => '',
                'desc_tip' => true,
            ),
        );
    }
    
    /**
     * Process payment
     */
    public function process_payment($order_id) {
        $order = wc_get_order($order_id);
        
        // Initialize API
        $api = new BOG_Payment_API($this->client_id, $this->client_secret, $this->testmode);
        
        // Prepare order data
        $order_data = $this->prepare_order_data($order);
        
        // Create order via API
        $result = $api->create_order($order_data);
        
        if ($result['success']) {
            // Save BOG order ID to order meta
            $order->update_meta_data('_bog_order_id', $result['order_id']);
            $order->save();
            
            // Add order note
            $order->add_order_note(
                /* translators: %s: BOG order ID */
                sprintf(__('BOG Payment initiated. Order ID: %s', 'bog-payment-gateway'), $result['order_id'])
            );
            
            // Return redirect
            return array(
                'result' => 'success',
                'redirect' => $result['redirect_url'],
            );
        } else {
            wc_add_notice(__('Payment error: ', 'bog-payment-gateway') . $result['error'], 'error');
            return array('result' => 'fail');
        }
    }
    
    /**
     * Prepare order data for API
     */
    private function prepare_order_data($order) {
        $callback_url = WC()->api_request_url('bog_payment_callback');
        
        $items = array();
        foreach ($order->get_items() as $item) {
            $product = $item->get_product();
            $unit_price = $item->get_subtotal() / $item->get_quantity();
            $items[] = array(
                'product_id' => (string) $product->get_id(),
                'description' => $item->get_name(),
                'quantity' => $item->get_quantity(),
                'unit_price' => $this->format_amount($unit_price),
                'total_price' => $this->format_amount($item->get_total()),
            );
        }
        
        // Add shipping as item if exists
        if ($order->get_shipping_total() > 0) {
            $items[] = array(
                'product_id' => 'shipping',
                'description' => __('Shipping', 'bog-payment-gateway'),
                'quantity' => 1,
                'unit_price' => $this->format_amount($order->get_shipping_total()),
                'total_price' => $this->format_amount($order->get_shipping_total()),
            );
        }
        
        return array(
            'callback_url' => $callback_url,
            'external_order_id' => (string) $order->get_id(),
            'purchase_units' => array(
                'currency' => $order->get_currency(),
                'total_amount' => $this->format_amount($order->get_total()),
                'basket' => $items,
            ),
            'redirect_urls' => array(
                'success' => $this->get_return_url($order),
                'fail' => wc_get_checkout_url(),
            ),
            'buyer' => array(
                'full_name' => $order->get_billing_first_name() . ' ' . $order->get_billing_last_name(),
                'email' => $order->get_billing_email(),
                'phone' => $order->get_billing_phone(),
            ),
        );
    }
    
    /**
     * Format amount for API (ensure proper decimal separator)
     */
    private function format_amount($amount) {
        // Convert to float and format with 2 decimals, using period as decimal separator
        return number_format((float) $amount, 2, '.', '');
    }
    
    /**
     * Handle callback from BOG
     */
    public function handle_callback() {
        // Get raw POST data
        $raw_data = file_get_contents('php://input');
        
        // Log raw callback data (for debugging)
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BOG Payment Callback - Raw Data: ' . $raw_data);
        }
        
        $callback_data = json_decode($raw_data, true);
        
        // BOG sends data wrapped in a 'body' object
        // Extract the actual order data from body if present
        if (isset($callback_data['body'])) {
            $order_data = $callback_data['body'];
        } else {
            $order_data = $callback_data;
        }
        
        // Log callback
        $this->log_callback($order_data);
        
        if (!$order_data || !isset($order_data['external_order_id'])) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('BOG Payment Callback Error: Missing external_order_id');
                error_log('Callback structure: ' . print_r($callback_data, true));
            }
            status_header(400);
            exit('Invalid callback data');
        }
        
        $order_id = $order_data['external_order_id'];
        $order = wc_get_order($order_id);
        
        if (!$order) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('BOG Payment Callback Error: Order #' . $order_id . ' not found');
            }
            status_header(404);
            exit('Order not found');
        }
        
        // Process callback with the extracted order data
        $result = BOG_Payment_Callback::get_instance()->process_callback($order_data, $order);
        
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('BOG Payment Callback Processed - Order #' . $order_id . ' - New Status: ' . $order->get_status());
        }
        
        status_header(200);
        exit('OK');
    }
    
    /**
     * Can refund order
     */
    public function can_refund_order($order) {
        return $order && $order->get_meta('_bog_order_id');
    }
    
    /**
     * Process refund
     */
    public function process_refund($order_id, $amount = null, $reason = '') {
        $order = wc_get_order($order_id);
        
        if (!$order) {
            return new WP_Error('error', __('Order not found', 'bog-payment-gateway'));
        }
        
        $bog_order_id = $order->get_meta('_bog_order_id');
        
        if (!$bog_order_id) {
            return new WP_Error('error', __('BOG Order ID not found', 'bog-payment-gateway'));
        }
        
        // Initialize API
        $api = new BOG_Payment_API($this->client_id, $this->client_secret, $this->testmode);
        
        // Process refund
        $result = $api->refund_payment($bog_order_id, $amount);
        
        if ($result['success']) {
            $order->add_order_note(
                sprintf(
                    /* translators: 1: Refund amount, 2: Refund reason */
                    __('Refund of %1$s processed via BOG. Reason: %2$s', 'bog-payment-gateway'), 
                    wc_price($amount), 
                    $reason
                )
            );
            return true;
        } else {
            return new WP_Error('error', $result['error']);
        }
    }
    
    /**
     * Log callback
     */
    private function log_callback($data) {
        global $wpdb;
        
        $table_name = $wpdb->prefix . 'bog_payment_logs';
        
        $wpdb->insert(
            $table_name,
            array(
                'order_id' => $data['order_id'] ?? '',
                'wc_order_id' => $data['external_order_id'] ?? null,
                'transaction_type' => 'callback',
                'request_data' => '',
                'response_data' => json_encode($data),
                'status' => $data['order_status']['key'] ?? 'unknown',
                'created_at' => current_time('mysql'),
            ),
            array('%s', '%d', '%s', '%s', '%s', '%s', '%s')
        );
    }
}
